package furny.swing.admin;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import javax.swing.AbstractAction;
import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import org.jdesktop.swingx.JXBusyLabel;
import org.jdesktop.swingx.JXLabel;

import furny.entities.Furniture;
import furny.entities.Tag;
import furny.furndb.FurnDBManager;
import furny.furndb.FurnitureUpdateListener;
import furny.furndb.SearchType;
import furny.swing.admin.tags.EditFurnitureTagsPanel;
import furny.swing.admin.viewer.IFurnitureViewer;

/**
 * This is an extended {@link JPanel} that contains a table displaying
 * furnitures.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class BrowserPanel extends JPanel implements FurnitureUpdateListener,
    IFurnitureBrowser {
  private final List<Long> furnIds = new ArrayList<Long>();
  private final Map<Long, Furniture> map = new HashMap<Long, Furniture>();

  private final JTable table;
  private final FurnitureTableModel tableModel;

  private final JTextField findField;
  private final JComboBox searchTypeComboBox;

  private final IFurnitureViewer viewer;

  /**
   * Instantiates a new browser panel.
   * 
   * @param viewer
   *          the viewer
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public BrowserPanel(final IFurnitureViewer viewer) {
    this.viewer = viewer;
    setLayout(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();

    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.NONE;
    constraints.weightx = 0.01d;
    constraints.weighty = 0.0d;
    constraints.gridwidth = 1;
    constraints.gridheight = 1;
    constraints.anchor = GridBagConstraints.EAST;

    final JLabel label = new JLabel("Find by");
    add(label, constraints);

    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 0.0d;
    constraints.gridx++;

    searchTypeComboBox = new JComboBox(SearchType.values());
    add(searchTypeComboBox, constraints);

    constraints.weightx = 1d;
    constraints.insets = new Insets(5, 5, 5, 0);
    constraints.gridx++;

    findField = new JTextField();
    findField.addActionListener(new ActionFind());

    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        findField.requestFocusInWindow();
      }
    });

    add(findField, constraints);

    constraints.weightx = 0.0d;
    constraints.insets = new Insets(5, 0, 5, 5);
    constraints.gridx++;

    final JButton button = new JButton(new ActionResetSearch());

    button.setMargin(new Insets(2, 5, 2, 5));

    add(button, constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.fill = GridBagConstraints.BOTH;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.anchor = GridBagConstraints.NORTH;
    constraints.gridwidth = 5;

    tableModel = new FurnitureTableModel();

    // overridden scrollRectToVisible to prevent autoscrolling
    table = new JTable(tableModel) {
      @Override
      public void scrollRectToVisible(final java.awt.Rectangle aRect) {
      }
    };
    table.setAutoscrolls(false);
    table.addMouseMotionListener(new MouseAdapter() {

      @Override
      public void mouseMoved(final MouseEvent e) {
        final Point hit = e.getPoint();
        final int hitColumn = table.columnAtPoint(hit);
        final int hitRow = table.rowAtPoint(hit);

        table.editCellAt(hitRow, hitColumn);
      }
    });

    table.setDefaultRenderer(Furniture.class, new FurnitureCellRenderer(null));

    final FurnitureCellRenderer editor = new FurnitureCellRenderer(this);
    editor.label.addMouseWheelListener(new MouseAdapter() {
      @Override
      public void mouseWheelMoved(final MouseWheelEvent e) {
        table.removeEditor();
        table.dispatchEvent(e);
      }
    });

    table.setDefaultEditor(Furniture.class, editor);

    table.setRowHeight(400);
    // table.setShowGrid(false);
    table.setRowSelectionAllowed(false);
    table.setColumnSelectionAllowed(false);
    table.setGridColor(getBackground());
    table.setTableHeader(null);
    table.setDragEnabled(false);

    add(new JScrollPane(table), constraints);

    constraints.gridy++;
    constraints.gridwidth = 5;
    constraints.fill = GridBagConstraints.NONE;
    constraints.weightx = 0d;
    constraints.weighty = 0d;
    constraints.anchor = GridBagConstraints.WEST;

    final JPanel buttonPanel = new JPanel();
    add(buttonPanel, constraints);

    buttonPanel.setLayout(new GridBagLayout());
    constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;

    buttonPanel.add(new JButton(new ActionRefresh()), constraints);

    constraints.gridx++;

    buttonPanel.add(new JButton(new ActionDeleteAll()), constraints);

    constraints.gridx++;

    buttonPanel.add(new JButton(new ActionAddTagsToAll()), constraints);

    addComponentListener(new ComponentAdapter() {
      @Override
      public void componentResized(final ComponentEvent e) {
        tableModel.fireTableStructureChanged();
      }
    });

    FurnDBManager.getInstance().addFurnitureUpdateListener(this);

    updateIds();
  }

  /**
   * Gets the current furniture ids from the {@link FurnDBManager}.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void updateIds() {
    FurnDBManager.getInstance().updateFurnitureIds();
  }

  /*
   * (non-Javadoc)
   * 
   * @see furny.furndb.admin.ISearchBar#searchByTag(java.lang.String)
   * 
   * @since 18.05.2011
   * 
   * @author stephan
   */
  @Override
  public void searchByTag(final String text) {
    searchTypeComboBox.setSelectedItem(SearchType.TAGS);
    findField.setText(text);

    find(true);
  }

  /**
   * Find furnitures by a text and search type.
   * 
   * @param type
   *          the type
   * @param text
   *          the text
   * @param equals
   *          if <code>true</code>, the search text must exactly match.
   * 
   * @see FurnDBManager#updateIdsByTagName(SearchType, String, boolean)
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void find(final SearchType type, final String text,
      final boolean equals) {
    if (text.isEmpty()) {
      updateIds();
    } else {
      FurnDBManager.getInstance().updateIdsByTagName(type, text, equals);
    }
  }

  /**
   * Triggers a search.
   * 
   * @param equals
   *          The text must equal if <code>true</code>
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void find(final boolean equals) {
    final SearchType type = (SearchType) searchTypeComboBox.getSelectedItem();
    find(type, findField.getText(), equals);
  }

  /**
   * Resets the search text field and results.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void resetSearch() {
    searchTypeComboBox.setSelectedItem(SearchType.ALL);
    findField.setText("");

    find(false);
  }

  @Override
  public void furnitureIdsUpdated(final List<Long> ids) {
    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {
        furnIds.clear();
        furnIds.addAll(ids);

        tableModel.fireTableDataChanged();
        table.removeEditor();
      }
    });
  }

  @Override
  public void furnitureUpdated(final Long id, final Furniture furniture) {
    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {
        map.put(id, furniture);
        table.repaint();

        table.removeEditor();
      }
    });
  }

  @Override
  public void furnitureDeleted(final Long id) {
    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {
        furnIds.remove(id);
        map.remove(id);

        tableModel.fireTableDataChanged();
        table.removeEditor();
      }
    });
  }

  /**
   * Table cell renderer for furnitures.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class FurnitureCellRenderer extends AbstractCellEditor implements
      TableCellRenderer, TableCellEditor {
    private final FurnitureLabel label;

    private final JLabel emptyLabel;

    private final JXBusyLabel busyLabel;

    /**
     * Instantiates a new furniture cell renderer.
     * 
     * @param searchBar
     *          the search bar
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public FurnitureCellRenderer(final IFurnitureBrowser searchBar) {
      label = new FurnitureLabel(searchBar, viewer);
      busyLabel = new JXBusyLabel(new Dimension(50, 50));
      busyLabel.setHorizontalAlignment(JXLabel.CENTER);
      busyLabel.setBusy(true);

      emptyLabel = new JLabel();
      emptyLabel.setBackground(getBackground());
      emptyLabel.setOpaque(true);
      emptyLabel.setBorder(BorderFactory.createEmptyBorder());
    }

    @Override
    public Component getTableCellRendererComponent(final JTable table,
        final Object value, final boolean isSelected, final boolean hasFocus,
        final int row, final int column) {

      if (value instanceof Furniture) {
        final Furniture furniture = (Furniture) value;
        label.setFurniture(furniture);
        return label;
      } else if (value instanceof Boolean) {
        if ((Boolean) value) {
          return busyLabel;
        } else {
          return emptyLabel;
        }
      }

      return null;
    }

    @Override
    public Object getCellEditorValue() {
      return null;
    }

    @Override
    public Component getTableCellEditorComponent(final JTable table,
        final Object value, final boolean isSelected, final int row,
        final int column) {
      return getTableCellRendererComponent(table,
          tableModel.getValueAt(row, column), isSelected, true, row, column);
    }
  }

  /**
   * Table model for furnitures.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class FurnitureTableModel extends DefaultTableModel {

    @Override
    public Class<?> getColumnClass(final int columnIndex) {
      return Furniture.class;
    }

    @Override
    public int getColumnCount() {
      if (table != null) {
        return table.getWidth() / 200;
      } else {
        return 3;
      }
    }

    @Override
    public int getRowCount() {
      return (int) Math.ceil(furnIds.size() / (double) getColumnCount());
    }

    @Override
    public void setValueAt(final Object aValue, final int row, final int column) {
    }

    @Override
    public Object getValueAt(final int row, final int column) {
      final int index = (row * getColumnCount()) + column;

      if (index < furnIds.size()) {
        final Long id = furnIds.get(index);

        final Furniture furniture = map.get(id);

        if (furniture != null) {
          return furniture;
        } else {
          FurnDBManager.getInstance().updateFurniture(id);

          return Boolean.TRUE;
        }

      }
      return Boolean.FALSE;
    }

    @Override
    public String getColumnName(final int column) {
      return "";
    }

    @Override
    public boolean isCellEditable(final int row, final int column) {
      return true;
    }
  }

  /**
   * Action to reset the search text field.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionResetSearch extends AbstractAction {

    /**
     * Instantiates a new action reset search.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionResetSearch() {
      super("X");
      putValue(SHORT_DESCRIPTION, "Reset the search");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      resetSearch();
    }
  }

  /**
   * Action to trigger a search.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionFind extends AbstractAction {

    /**
     * Instantiates a new action find.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionFind() {
      super("Find");
      putValue(SHORT_DESCRIPTION, "Find furnitures");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      find(false);
    }
  }

  /**
   * Action to refresh the furniture ids.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionRefresh extends AbstractAction {

    /**
     * Instantiates a new action refresh.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionRefresh() {
      super("Refresh");
      putValue(SHORT_DESCRIPTION, "Refresh furnitures from database");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      furnIds.clear();
      map.clear();
      resetSearch();
    }
  }

  /**
   * Action to delete all visible furnitures.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionDeleteAll extends AbstractAction {

    /**
     * Instantiates a new action delete all.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionDeleteAll() {
      super("Delete all furnitures");
      putValue(SHORT_DESCRIPTION, "Delete all displayed furnitures");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      int n = JOptionPane.showConfirmDialog(null,
          "Are you sure to delete furnitures " + furnIds + "?",
          "Delete Furniture", JOptionPane.YES_NO_OPTION,
          JOptionPane.QUESTION_MESSAGE);

      if (n == JOptionPane.YES_OPTION) {
        n = JOptionPane
            .showConfirmDialog(
                null,
                "Are you really sure to delete all displayed furnitures?\nDeletion can not be undone!",
                "Delete Furniture", JOptionPane.YES_NO_OPTION,
                JOptionPane.WARNING_MESSAGE);

        if (n == JOptionPane.YES_OPTION) {
          for (final Long id : furnIds) {
            FurnDBManager.getInstance().deleteFurniture(id);
          }

          resetSearch();
          viewer.setFurniture(null);
        }
      }
    }
  }

  /**
   * Action to add tags to all visible furnitures.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionAddTagsToAll extends AbstractAction {

    /**
     * Instantiates a new action add tags to all.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionAddTagsToAll() {
      super("Add tags to all");
      putValue(SHORT_DESCRIPTION,
          "Add multiple tags to all displayed furnitures");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final EditFurnitureTagsPanel tp = new EditFurnitureTagsPanel();

      // final List<Tag> existingTags = new
      // ArrayList<Tag>(furniture.getMetaData()
      // .getTags());
      // tp.addTags(existingTags);

      final int n = JOptionPane.showConfirmDialog(null, tp,
          "Edit tags for furniture", JOptionPane.OK_CANCEL_OPTION,
          JOptionPane.PLAIN_MESSAGE);

      if (n == JOptionPane.OK_OPTION) {
        final List<Tag> tags = tp.getSelectedTags();

        for (final Long id : furnIds) {
          final Furniture furniture = FurnDBManager.getInstance().getFurniture(
              id);
          FurnDBManager.getInstance().addFurnitureTags(furniture,
              new TreeSet<Tag>(tags));
        }
      }
    }
  }
}
